<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\IAEvaluacionService;

class EvaluacionController extends Controller
{
    protected $iaService;

    public function __construct(IAEvaluacionService $iaService)
    {
        $this->iaService = $iaService;
    }

    /**
     * Evalúa una pregunta individual con IA
     */
    public function evaluarPregunta(Request $request)
    {
        $request->validate([
            'pregunta' => 'required|string',
            'respuesta_correcta' => 'required|string',
            'respuesta_estudiante' => 'required|string',
            'materia' => 'required|string',
            'curso' => 'required|string',
        ]);

        $retroalimentacion = $this->iaService->generarRetroalimentacion(
            $request->pregunta,
            $request->respuesta_correcta,
            $request->respuesta_estudiante,
            $request->materia,
            $request->curso
        );

        return response()->json([
            'success' => true,
            'retroalimentacion' => $retroalimentacion
        ]);
    }

    /**
     * Evalúa una prueba completa
     */
    public function evaluarPrueba(Request $request)
    {
        $request->validate([
            'prueba_id' => 'required|integer',
            'respuestas' => 'required|array',
            'materia' => 'required|string',
            'curso' => 'required|string',
        ]);

        $resultados = [];
        $respuestasDetalladas = [];

        foreach ($request->respuestas as $respuesta) {
            if (!$respuesta['correcta']) {
                // Solo generar retroalimentación para respuestas incorrectas
                $retro = $this->iaService->generarRetroalimentacion(
                    $respuesta['pregunta'],
                    $respuesta['respuesta_correcta'],
                    $respuesta['respuesta_estudiante'],
                    $request->materia,
                    $request->curso
                );

                $respuestasDetalladas[] = array_merge($respuesta, [
                    'retroalimentacion' => $retro
                ]);
            } else {
                $respuestasDetalladas[] = $respuesta;
            }
        }

        // Generar resumen general
        $resumen = $this->iaService->generarResumenPrueba(
            $respuestasDetalladas,
            $request->materia,
            $request->curso
        );

        return response()->json([
            'success' => true,
            'respuestas' => $respuestasDetalladas,
            'resumen' => $resumen,
            'total_preguntas' => count($request->respuestas),
            'correctas' => count(array_filter($request->respuestas, fn($r) => $r['correcta'])),
            'incorrectas' => count(array_filter($request->respuestas, fn($r) => !$r['correcta']))
        ]);
    }
}