<?php

namespace App\Services;

use OpenAI;

class IAEvaluacionService
{
    private $client;
    
    // Mapeo de materias y temas con sus páginas de estudio
    private $materiasContenido = [
        'matematicas' => [
            'algebra' => '/curso/matematicas/algebra',
            'geometria' => '/curso/matematicas/geometria',
            'calculo' => '/curso/matematicas/calculo',
            'trigonometria' => '/curso/matematicas/trigonometria',
            'ecuaciones' => '/curso/matematicas/ecuaciones',
        ],
        'lenguaje' => [
            'gramatica' => '/curso/lenguaje/gramatica',
            'ortografia' => '/curso/lenguaje/ortografia',
            'literatura' => '/curso/lenguaje/literatura',
            'comprension' => '/curso/lenguaje/comprension',
            'redaccion' => '/curso/lenguaje/redaccion',
        ],
        'ciencias' => [
            'biologia' => '/curso/ciencias/biologia',
            'quimica' => '/curso/ciencias/quimica',
            'fisica' => '/curso/ciencias/fisica',
            'anatomia' => '/curso/ciencias/anatomia',
        ],
        'historia' => [
            'antigua' => '/curso/historia/antigua',
            'medieval' => '/curso/historia/medieval',
            'moderna' => '/curso/historia/moderna',
            'chile' => '/curso/historia/chile',
        ],
        'ingles' => [
            'gramatica' => '/curso/ingles/gramatica',
            'vocabulario' => '/curso/ingles/vocabulario',
            'conversacion' => '/curso/ingles/conversacion',
        ]
    ];

    public function __construct()
    {
        $this->client = OpenAI::client(config('services.openai.key'));
    }

    /**
     * Genera retroalimentación inteligente para respuestas incorrectas
     */
    public function generarRetroalimentacion($pregunta, $respuestaCorrecta, $respuestaEstudiante, $materia, $curso)
    {
        try {
            $prompt = $this->construirPrompt($pregunta, $respuestaCorrecta, $respuestaEstudiante, $materia, $curso);
            
            $response = $this->client->chat()->create([
                'model' => config('services.openai.model', 'gpt-4o-mini'),
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'Eres un tutor educativo experto y empático. Tu objetivo es ayudar a estudiantes a entender sus errores de forma constructiva y motivadora. Debes proporcionar explicaciones claras, identificar el concepto que deben repasar y sugerir recursos específicos.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.7,
                'max_tokens' => 500
            ]);

            $contenido = $response->choices[0]->message->content;
            
            // Procesar la respuesta para extraer información estructurada
            return $this->procesarRespuestaIA($contenido, $materia);

        } catch (\Exception $e) {
            \Log::error('Error en IA: ' . $e->getMessage());
            return $this->respuestaFallback($pregunta, $respuestaCorrecta, $materia);
        }
    }

    /**
     * Construye el prompt para la IA
     */
    private function construirPrompt($pregunta, $respuestaCorrecta, $respuestaEstudiante, $materia, $curso)
    {
        return "
Contexto:
- Materia: {$materia}
- Curso: {$curso}

Pregunta del examen:
{$pregunta}

Respuesta correcta:
{$respuestaCorrecta}

Respuesta del estudiante:
{$respuestaEstudiante}

Por favor proporciona:
1. EXPLICACION: Una explicación breve (2-3 oraciones) de por qué la respuesta del estudiante es incorrecta
2. CONCEPTO: El concepto o tema específico que el estudiante debe repasar (una o dos palabras clave)
3. SUGERENCIA: Un consejo motivador para mejorar (1 oración)

Formato de respuesta:
EXPLICACION: [tu explicación aquí]
CONCEPTO: [concepto clave]
SUGERENCIA: [tu sugerencia aquí]
";
    }

    /**
     * Procesa la respuesta de la IA y extrae información estructurada
     */
    private function procesarRespuestaIA($contenido, $materia)
    {
        // Extraer secciones usando regex
        preg_match('/EXPLICACION:\s*(.+?)(?=CONCEPTO:|$)/s', $contenido, $explicacion);
        preg_match('/CONCEPTO:\s*(.+?)(?=SUGERENCIA:|$)/s', $contenido, $concepto);
        preg_match('/SUGERENCIA:\s*(.+?)$/s', $contenido, $sugerencia);

        $conceptoLimpio = isset($concepto[1]) ? trim(strtolower($concepto[1])) : '';
        
        // Buscar la página de estudio más relevante
        $paginaEstudio = $this->buscarPaginaEstudio($conceptoLimpio, $materia);

        return [
            'explicacion' => isset($explicacion[1]) ? trim($explicacion[1]) : 'La respuesta seleccionada no es correcta.',
            'concepto' => $conceptoLimpio ?: 'concepto general',
            'sugerencia' => isset($sugerencia[1]) ? trim($sugerencia[1]) : 'Revisa el material del curso y vuelve a intentarlo.',
            'pagina_estudio' => $paginaEstudio,
            'url_completa' => config('app.url') . $paginaEstudio
        ];
    }

    /**
     * Busca la página de estudio más relevante según el concepto
     */
    private function buscarPaginaEstudio($concepto, $materia)
    {
        $materiaKey = strtolower($materia);
        
        // Si la materia existe en nuestro mapeo
        if (isset($this->materiasContenido[$materiaKey])) {
            // Buscar coincidencia en los temas
            foreach ($this->materiasContenido[$materiaKey] as $tema => $url) {
                if (stripos($concepto, $tema) !== false || stripos($tema, $concepto) !== false) {
                    return $url;
                }
            }
            
            // Si no hay coincidencia específica, retornar la primera página de la materia
            return reset($this->materiasContenido[$materiaKey]);
        }
        
        // Página de estudio por defecto
        return '/cursos';
    }

    /**
     * Respuesta de fallback si la IA falla
     */
    private function respuestaFallback($pregunta, $respuestaCorrecta, $materia)
    {
        $materiaKey = strtolower($materia);
        $paginaEstudio = isset($this->materiasContenido[$materiaKey]) 
            ? reset($this->materiasContenido[$materiaKey])
            : '/cursos';

        return [
            'explicacion' => 'La respuesta que seleccionaste no es correcta. Te recomendamos revisar el material del curso.',
            'concepto' => 'tema general',
            'sugerencia' => 'Revisa los apuntes y ejemplos de esta sección antes de volver a intentarlo.',
            'pagina_estudio' => $paginaEstudio,
            'url_completa' => config('app.url') . $paginaEstudio
        ];
    }

    /**
     * Analiza múltiples respuestas y genera un resumen
     */
    public function generarResumenPrueba($respuestas, $materia, $curso)
    {
        $incorrectas = array_filter($respuestas, function($r) {
            return !$r['correcta'];
        });

        if (empty($incorrectas)) {
            return [
                'mensaje' => '¡Excelente trabajo! Has respondido todas las preguntas correctamente.',
                'areas_mejorar' => [],
                'recomendacion_general' => 'Continúa con el buen trabajo y sigue practicando.'
            ];
        }

        // Identificar patrones en los errores
        $conceptos = array_column($incorrectas, 'concepto');
        $conceptosUnicos = array_unique($conceptos);

        return [
            'mensaje' => count($incorrectas) === 1 
                ? 'Tuviste un error en la prueba.' 
                : 'Tuviste ' . count($incorrectas) . ' errores en la prueba.',
            'areas_mejorar' => $conceptosUnicos,
            'recomendacion_general' => 'Te recomendamos enfocarte en: ' . implode(', ', $conceptosUnicos)
        ];
    }
}