/**
 * Sistema de Evaluación con IA
 * Integración con retroalimentación inteligente
 */

class EvaluacionIA {
    constructor() {
        this.apiUrl = window.location.origin;
        this.csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';
    }

    /**
     * Evalúa una prueba completa con IA
     */
    async evaluarPrueba(pruebaId, respuestas, materia, curso) {
        try {
            // Mostrar loading
            this.mostrarLoading('Evaluando tus respuestas con IA...');

            const response = await fetch(`${this.apiUrl}/api/evaluar-prueba`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.csrfToken,
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    prueba_id: pruebaId,
                    respuestas: respuestas,
                    materia: materia,
                    curso: curso
                })
            });

            if (!response.ok) {
                throw new Error('Error en la evaluación');
            }

            const data = await response.json();
            this.ocultarLoading();
            
            return data;

        } catch (error) {
            console.error('Error:', error);
            this.ocultarLoading();
            return this.evaluacionFallback(respuestas);
        }
    }

    /**
     * Evaluación de fallback si la IA no responde
     */
    evaluacionFallback(respuestas) {
        return {
            success: true,
            respuestas: respuestas.map(r => ({
                ...r,
                retroalimentacion: r.correcta ? null : {
                    explicacion: 'La respuesta seleccionada no es correcta. Revisa el material del curso.',
                    concepto: 'tema general',
                    sugerencia: 'Te recomendamos repasar esta sección.',
                    pagina_estudio: '/cursos',
                    url_completa: window.location.origin + '/cursos'
                }
            })),
            resumen: {
                mensaje: `Completaste la prueba con ${respuestas.filter(r => r.correcta).length} respuestas correctas.`,
                areas_mejorar: [],
                recomendacion_general: 'Sigue estudiando y practicando.'
            },
            total_preguntas: respuestas.length,
            correctas: respuestas.filter(r => r.correcta).length,
            incorrectas: respuestas.filter(r => !r.correcta).length
        };
    }

    /**
     * Muestra los resultados con retroalimentación de IA
     */
    mostrarResultadosConIA(resultado) {
        const { respuestas, resumen, correctas, incorrectas, total_preguntas } = resultado;
        const score = Math.round((correctas / total_preguntas) * 100);

        // Limpiar sección de resultados
        const container = document.getElementById('testResultSection');
        if (!container) return;

        container.innerHTML = `
            <div class="test-container">
                <!-- Resumen General -->
                <div class="result-card">
                    <div class="result-score">${score}%</div>
                    <div class="result-message">${this.getMensajePorScore(score)}</div>
                    <div class="result-details">
                        <div class="result-detail-item">
                            <h4>${correctas}</h4>
                            <p>Correctas</p>
                        </div>
                        <div class="result-detail-item">
                            <h4>${incorrectas}</h4>
                            <p>Incorrectas</p>
                        </div>
                        <div class="result-detail-item">
                            <h4>${total_preguntas}</h4>
                            <p>Total</p>
                        </div>
                    </div>
                    
                    ${resumen.areas_mejorar.length > 0 ? `
                        <div class="alert alert-info mt-4">
                            <i class="fas fa-lightbulb"></i>
                            <strong>Áreas a mejorar:</strong> ${resumen.areas_mejorar.join(', ')}
                        </div>
                    ` : ''}
                </div>

                <!-- Retroalimentación Detallada -->
                ${incorrectas > 0 ? this.generarRetroalimentacionDetallada(respuestas) : ''}

                <button class="btn-primary-custom mt-4" onclick="backToDashboard()">
                    <i class="fas fa-home"></i> Volver al Dashboard
                </button>
            </div>
        `;

        // Mostrar sección
        document.getElementById('testTakingSection').style.display = 'none';
        container.style.display = 'block';
    }

    /**
     * Genera HTML para retroalimentación detallada
     */
    generarRetroalimentacionDetallada(respuestas) {
        const incorrectas = respuestas.filter(r => !r.correcta && r.retroalimentacion);
        
        if (incorrectas.length === 0) return '';

        return `
            <div class="content-section mt-4">
                <div class="section-header">
                    <h3><i class="fas fa-graduation-cap"></i> Retroalimentación con IA</h3>
                </div>
                
                ${incorrectas.map((resp, index) => `
                    <div class="feedback-card mb-4">
                        <div class="feedback-header">
                            <span class="feedback-number">Pregunta ${index + 1}</span>
                            <span class="feedback-status error">
                                <i class="fas fa-times-circle"></i> Incorrecta
                            </span>
                        </div>
                        
                        <div class="feedback-question">
                            <strong>Pregunta:</strong> ${resp.pregunta}
                        </div>
                        
                        <div class="feedback-answers">
                            <div class="answer-item wrong">
                                <i class="fas fa-times"></i>
                                <strong>Tu respuesta:</strong> ${resp.respuesta_estudiante}
                            </div>
                            <div class="answer-item correct">
                                <i class="fas fa-check"></i>
                                <strong>Respuesta correcta:</strong> ${resp.respuesta_correcta}
                            </div>
                        </div>
                        
                        ${resp.retroalimentacion ? `
                            <div class="ia-feedback">
                                <div class="ia-badge">
                                    <i class="fas fa-robot"></i> Retroalimentación con IA
                                </div>
                                
                                <div class="feedback-section">
                                    <i class="fas fa-info-circle"></i>
                                    <strong>¿Por qué es incorrecta?</strong>
                                    <p>${resp.retroalimentacion.explicacion}</p>
                                </div>
                                
                                <div class="feedback-section">
                                    <i class="fas fa-book"></i>
                                    <strong>Concepto a repasar:</strong>
                                    <p>${resp.retroalimentacion.concepto}</p>
                                </div>
                                
                                <div class="feedback-section">
                                    <i class="fas fa-lightbulb"></i>
                                    <strong>Sugerencia:</strong>
                                    <p>${resp.retroalimentacion.sugerencia}</p>
                                </div>
                                
                                <a href="${resp.retroalimentacion.pagina_estudio}" 
                                   class="btn-study-link"
                                   target="_blank">
                                    <i class="fas fa-external-link-alt"></i>
                                    Ir a la página de estudio
                                </a>
                            </div>
                        ` : ''}
                    </div>
                `).join('')}
            </div>
        `;
    }

    /**
     * Obtiene mensaje según el puntaje
     */
    getMensajePorScore(score) {
        if (score >= 90) return '¡Excelente trabajo! 🌟';
        if (score >= 70) return '¡Muy bien! 👏';
        if (score >= 50) return '¡Buen esfuerzo! 💪';
        return 'Sigue practicando 📚';
    }

    /**
     * Muestra indicador de carga
     */
    mostrarLoading(mensaje = 'Cargando...') {
        const loadingHtml = `
            <div id="iaLoadingOverlay" style="
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.7);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 9999;
            ">
                <div style="
                    background: white;
                    padding: 30px 50px;
                    border-radius: 15px;
                    text-align: center;
                ">
                    <div class="spinner-border text-primary mb-3" role="status">
                        <span class="visually-hidden">Cargando...</span>
                    </div>
                    <h5>${mensaje}</h5>
                </div>
            </div>
        `;
        document.body.insertAdjacentHTML('beforeend', loadingHtml);
    }

    /**
     * Oculta indicador de carga
     */
    ocultarLoading() {
        const overlay = document.getElementById('iaLoadingOverlay');
        if (overlay) overlay.remove();
    }
}

// Instancia global
window.evaluacionIA = new EvaluacionIA();