(function () {
  ("use strict");

  // Cache DOM elements
  const DOM = {
    body: document.body,
    header: document.querySelector("#header"),
    mobileNavToggle: document.querySelector(".mobile-nav-toggle"),
    navmenu: document.querySelector("#navmenu"),
    preloader: document.querySelector("#preloader"),
    scrollTop: document.querySelector(".scroll-top"),
  };

  // Utility functions
  const utils = {
    debounce(func, wait) {
      let timeout;
      return function executedFunction(...args) {
        const later = () => {
          clearTimeout(timeout);
          func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
      };
    },

    hasClass(element, className) {
      return element?.classList.contains(className);
    },
  };

  // Header scroll functionality
  function toggleScrolled() {
    if (!DOM.header) return;

    const hasScrollClasses = [
      "scroll-up-sticky",
      "sticky-top",
      "fixed-top",
    ].some((cls) => DOM.header.classList.contains(cls));

    if (!hasScrollClasses) return;

    DOM.body.classList.toggle("scrolled", window.scrollY > 100);
  }

  // Mobile navigation
  const mobileNav = {
    toggle() {
      if (!DOM.mobileNavToggle) return;

      DOM.body.classList.toggle("mobile-nav-active");
      DOM.mobileNavToggle.classList.toggle("bi-list");
      DOM.mobileNavToggle.classList.toggle("bi-x");
    },

    close() {
      if (DOM.body.classList.contains("mobile-nav-active")) {
        this.toggle();
      }
    },

    init() {
      if (!DOM.mobileNavToggle) return;

      DOM.mobileNavToggle.addEventListener("click", this.toggle.bind(this));

      // Close on navigation click
      if (DOM.navmenu) {
        DOM.navmenu.addEventListener("click", (e) => {
          if (e.target.matches("a")) {
            this.close();
          }
        });
      }
    },
  };

  // Dropdown functionality
  function initDropdowns() {
    const dropdownToggles = document.querySelectorAll(
      ".navmenu .toggle-dropdown"
    );

    dropdownToggles.forEach((toggle) => {
      toggle.addEventListener("click", function (e) {
        e.preventDefault();
        e.stopImmediatePropagation();

        const parent = this.parentNode;
        const dropdown = parent.nextElementSibling;

        parent.classList.toggle("active");
        dropdown?.classList.toggle("dropdown-active");
      });
    });
  }

  // Scroll to top functionality
  const scrollToTop = {
    toggle() {
      if (!DOM.scrollTop) return;
      DOM.scrollTop.classList.toggle("active", window.scrollY > 100);
    },

    scrollUp() {
      window.scrollTo({
        top: 0,
        behavior: "smooth",
      });
    },

    init() {
      if (!DOM.scrollTop) return;

      DOM.scrollTop.addEventListener("click", (e) => {
        e.preventDefault();
        this.scrollUp();
      });
    },
  };

  // AOS initialization
  function initAOS() {
    if (typeof AOS !== "undefined") {
      AOS.init({
        duration: 600,
        easing: "ease-in-out",
        once: true,
        mirror: false,
      });
    }
  }

  // Carousel indicators
  function generateCarouselIndicators() {
    document.querySelectorAll(".carousel-indicators").forEach((indicator) => {
      const carousel = indicator.closest(".carousel");
      const items = carousel.querySelectorAll(".carousel-item");

      const indicatorsHTML = Array.from(items)
        .map((_, index) => {
          const activeClass = index === 0 ? ' class="active"' : "";
          return `<li data-bs-target="#${carousel.id}" data-bs-slide-to="${index}"${activeClass}></li>`;
        })
        .join("");

      indicator.innerHTML = indicatorsHTML;
    });
  }

  // GLightbox initialization
  function initGLightbox() {
    if (typeof GLightbox !== "undefined") {
      GLightbox({
        selector: ".glightbox",
      });
    }
  }

  // Isotope layout
  function initIsotope() {
    document.querySelectorAll(".isotope-layout").forEach((item) => {
      const container = item.querySelector(".isotope-container");
      if (!container) return;

      const config = {
        itemSelector: ".isotope-item",
        layoutMode: item.getAttribute("data-layout") || "masonry",
        filter: item.getAttribute("data-default-filter") || "*",
        sortBy: item.getAttribute("data-sort") || "original-order",
      };

      let isotope;

      if (
        typeof imagesLoaded !== "undefined" &&
        typeof Isotope !== "undefined"
      ) {
        imagesLoaded(container, () => {
          isotope = new Isotope(container, config);
        });

        // Filter functionality
        item.querySelectorAll(".isotope-filters li").forEach((filter) => {
          filter.addEventListener("click", function () {
            // Update active filter
            item
              .querySelector(".filter-active")
              ?.classList.remove("filter-active");
            this.classList.add("filter-active");

            // Apply filter
            isotope?.arrange({
              filter: this.getAttribute("data-filter"),
            });

            // Reinit AOS if available
            if (typeof initAOS === "function") {
              initAOS();
            }
          });
        });
      }
    });
  }

  // Swiper initialization
  function initSwiper() {
    if (typeof Swiper === "undefined") return;

    document.querySelectorAll(".init-swiper").forEach((element) => {
      const configElement = element.querySelector(".swiper-config");
      if (!configElement) return;

      try {
        const config = JSON.parse(configElement.innerHTML.trim());

        if (element.classList.contains("swiper-tab")) {
          initSwiperWithCustomPagination(element, config);
        } else {
          new Swiper(element, config);
        }
      } catch (error) {
        console.error("Error parsing Swiper config:", error);
      }
    });
  }

  // Hash link scrolling
  function handleHashLinkScrolling() {
    if (!window.location.hash) return;

    const target = document.querySelector(window.location.hash);
    if (!target) return;

    setTimeout(() => {
      const scrollMarginTop =
        parseInt(getComputedStyle(target).scrollMarginTop) || 0;
      window.scrollTo({
        top: target.offsetTop - scrollMarginTop,
        behavior: "smooth",
      });
    }, 100);
  }

  // Navigation scrollspy
  const scrollSpy = {
    links: [],

    init() {
      this.links = Array.from(document.querySelectorAll(".navmenu a"))
        .filter((link) => link.hash)
        .map((link) => ({
          element: link,
          target: document.querySelector(link.hash),
          hash: link.hash,
        }))
        .filter((item) => item.target);
    },

    update() {
      const position = window.scrollY + 200;

      // Remove all active classes first
      document.querySelectorAll(".navmenu a.active").forEach((link) => {
        link.classList.remove("active");
      });

      // Find and activate current section
      this.links.forEach(({ element, target }) => {
        const isInView =
          position >= target.offsetTop &&
          position <= target.offsetTop + target.offsetHeight;

        if (isInView) {
          element.classList.add("active");
        }
      });
    },
  };

  // Event listeners with debouncing for performance
  const debouncedToggleScrolled = utils.debounce(toggleScrolled, 10);
  const debouncedScrollTopToggle = utils.debounce(scrollToTop.toggle, 10);
  const debouncedScrollSpy = utils.debounce(
    scrollSpy.update.bind(scrollSpy),
    10
  );

  // Initialize everything on load
  function init() {
    // Remove preloader
    DOM.preloader?.remove();

    // Initialize components
    mobileNav.init();
    initDropdowns();
    scrollToTop.init();
    scrollSpy.init();

    // Initialize external libraries
    initAOS();
    generateCarouselIndicators();
    initGLightbox();
    initIsotope();
    initSwiper();

    // Handle hash links
    handleHashLinkScrolling();

    // Initial state
    toggleScrolled();
    scrollToTop.toggle();
    scrollSpy.update();
  }

  // Event listeners
  document.addEventListener("scroll", () => {
    debouncedToggleScrolled();
    debouncedScrollTopToggle();
    debouncedScrollSpy();
  });

  window.addEventListener("load", init);
})();
